/*
	DeFreq plugin for Avisynth 2.5 - Interference frequency remover

	Copyright (C) 2004 A.G.Balakhnin aka Fizick, bag@hotmail.ru, http://bag.hotmail.ru

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation.

	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with this program; if not, write to the Free Software
	Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

	Plugin uses external FFTW library version 3 (http://www.fftw.org)
	as Windows binary DLL (compiled with gcc under MinGW by Alessio Massaro),
	which support for threads and have AMD K7 (3dNow!) support in addition to SSE/SSE2.
	It may be downloaded from ftp://ftp.fftw.org/pub/fftw/fftw3win32mingw.zip
	You must put FFTW3.DLL file from this package to some directory in path
	(for example, C:\WINNT).
	
	Version 0.1, 22 September 2004 (initial, not public)
	Version 0.2, 26 September 2004 - add second  search window, set other planes to 128
	Version 0.3, 28 September 2004 - change cleaning method to adaptive in window,
		add third and fourth search windows, change default sharp=50.0 (was 100.0 really)
	Version 0.4, 30 September 2004 - change fx,fy,dx,dy,cutx,cuty parameters to percent of size (float)
	Version 0.4.1, 01 October 2004 - fixed bug with init
	Version 0.5, 02 October 2004 - add show sample frequency stripes, Measure mode, some bug fixed
	Version 0.5.0.1, 06 October 2004 - add Russian doc (not public)
	Version 0.5.1, 07 October 2004 - more fast processing of frames which are not needed in cleaning,
	       source released under GPL.
	Version 0.5.2, 7 November 2004 - fixed strange bug for separated fields. Thanks to johnmeyer for report. 
	Version 0.5.3, 8 November 2004 - finally fixed bug for separated fields. Thanks to johnmeyer for report. 
	Version 0.5.4, 3 December 2004 - Fixed bug for second peak for YUY2. Thanks to oledoe for report. 
	Version 0.6, August 2, 2005 - added temporal average in show mode (thanks to lumier for idea), 
		changed show parameter to integer type; reorganized view of the show window; reorganize a code. 
*/

#if (_MSC_VER >= 1300) && (WINVER < 0x0500)
//VC7 or later, building with pre-VC7 runtime libraries
extern "C" long _ftol( double ); //defined by VC6 C libs
extern "C" long _ftol2( double dblSource ) { return _ftol( dblSource ); }
#endif

#include "windows.h"
#include "avisynth.h"
#include "math.h"
#include "fftw3.h"
#include "info.h"


//-------------------------------------------------------------------------------------------
class DeFreq : public GenericVideoFilter {   
  // DeFreq defines the name of your filter class. 
  // This name is only used internally, and does not affect the name of your filter or similar.
  // This filter extends GenericVideoFilter, which incorporates basic functionality.
  // All functions present in the filter must also be present here.

	//  parameters
	float fx;
	float fy;
	float dx;
	float dy;
	float sharp;
	float fx2;
	float fy2;
	float dx2;
	float dy2;
	float sharp2;
	float fx3;
	float fy3;
	float dx3;
	float dy3;
	float sharp3;
	float fx4;
	float fy4;
	float dx4;
	float dy4;
	float sharp4;
	float cutx;
	float cuty;
	int plane;
	int show;
	bool info;
	bool measure;

	// additional parameterss
	float *in;
	fftwf_complex *out;
	fftwf_plan plan, plani;
	int nx, ny;
	int outwidth;

	char messagebuf[32];
	float *psd; // power spectral density (average) - added in v.0.6
	int naverage; // number of avaraged frames

	void DeFreqYV12(BYTE *srcp, int height, int width, int pitch,
		float * fxPeak, float * fyPeak, float *sharpPeak, float * fxPeak2, float * fyPeak2, float *sharpPeak2,
		float * fxPeak3, float * fyPeak3, float *sharpPeak3, float * fxPeak4, float * fyPeak4, float *sharpPeak4);
	void DeFreqYUY2(int plane, BYTE *srcp, int height, int width, int pitch,
		float * fxPeak, float * fyPeak, float *sharpPeak,  float * fxPeak2, float * fyPeak2, float *sharpPeak2,
		float * fxPeak3, float * fyPeak3, float *sharpPeak3, float * fxPeak4, float * fyPeak4, float *sharpPeak4);
	void SearchPeak(float * psd, int outwidth, int height, float fx, float fy, float dx, float dy,
		float * fxPeak, float * fyPeak, float *sharpPeak);
	void CleanPeak(fftwf_complex * out, int outwidth, int height, float fxPeak, float fyPeak);
	void CleanWindow(fftwf_complex * out, int outwidth, int height, float fx, float fy, float dx, float dy,
		float fxPeak, float fyPeak, float sharpPeak);
	void CleanHigh(fftwf_complex * outp, int outwidth, int height, float cutx, float cuty);
	void GetFFT2minmax(float * psd, int outwidth, int height, float *fft2min, float * fft2max);
	void DrawSearchBox(float * psd, int outwidth, int height, float fx, float fy, float dx, float dy, float fftval);
	void FrequencySwitchOn(fftwf_complex * outp, int outwidth, int height, float fx, float fy, float setvalue); // added in v.0.5

public:
  // This defines that these functions are present in your class.
  // These functions must be that same as those actually implemented.
  // Since the functions are "public" they are accessible to other classes.
  // Otherwise they can only be called from functions within the class itself.

	DeFreq(PClip _child, float _fx, float _fy, float _dx, float _dy, float _sharp,
		float _fx2, float _fy2,  float _dx2, float _dy2, float _sharp2,
		float _fx3, float _fy3,  float _dx3, float _dy3, float _sharp3,
		float _fx4, float _fy4,  float _dx4, float _dy4, float _sharp4,
		float _cutx, float _cuty, int _plane, int _show, bool _info, bool _measure, IScriptEnvironment* env);
  // This is the constructor. It does not return any value, and is always used, 
  //  when an instance of the class is created.
  // Since there is no code in this, this is the definition.

  ~DeFreq();
  // The is the destructor definition. This is called when the filter is destroyed.


	PVideoFrame __stdcall GetFrame(int n, IScriptEnvironment* env);
  // This is the function that AviSynth calls to get a given frame.
  // So when this functions gets called, the filter is supposed to return frame n.
};

// The following is the implementation 
// of the defined functions.

//Here is the acutal constructor code used
DeFreq::DeFreq(PClip _child, float _fx, float _fy, float _dx, float _dy, float _sharp,
		float _fx2, float _fy2,  float _dx2, float _dy2, float _sharp2,
		float _fx3, float _fy3,  float _dx3, float _dy3, float _sharp3,
		float _fx4, float _fy4,  float _dx4, float _dy4, float _sharp4,
		float _cutx, float _cuty, int _plane, int _show, bool _info, bool _measure, IScriptEnvironment* env):

	GenericVideoFilter(_child), fx(_fx), fy(_fy),  dx(_dx), dy(_dy), sharp(_sharp),
		fx2(_fx2), fy2(_fy2), dx2(_dx2), dy2(_dy2), sharp2(_sharp2),
		fx3(_fx3), fy3(_fy3), dx3(_dx3), dy3(_dy3), sharp3(_sharp3),
		fx4(_fx4), fy4(_fy4), dx4(_dx4), dy4(_dy4), sharp4(_sharp4),
		cutx(_cutx), cuty(_cuty), plane(_plane), show(_show), info(_info), measure(_measure) {
  // This is the implementation of the constructor.
  // The child clip (source clip) is inherited by the GenericVideoFilter,
  //  where the following variables gets defined:
  //   PClip child;   // Contains the source clip.
  //   VideoInfo vi;  // Contains videoinfo on the source clip.

		if ( fx<0 || fx>100 || fx2<0 || fx2>100 || fx3<0 || fx3>100 || fx4<0 || fx4>100 )
			env->ThrowError("DeFreq: fx,fx2,fx3,fx4 must be from 0.0 to 100.0%% !");
		if ( fy<-100 || fy>100 || fy2<-100 || fy2>100 || fy3<-100 || fy3>100 || fy4<-100 || fy4>100 )
			env->ThrowError("DeFreq: fy,fy2,fy3,f4 must be from -100.0 to 100.0%% !");
		if ( cutx<0 || cutx>200 )
			env->ThrowError("DeFreq: cutx must be from 0.0 to 300.0%% !");
		if ( cuty<0 || cuty>200 )
			env->ThrowError("DeFreq: cuty must be from 0.0 to 300.0%% !");
		if ( dx<0 || dx>50 || dx2<0 || dx2>50 || dx3<0 || dx3>50 || dx4<0 || dx4>50 )
			env->ThrowError("DeFreq: dx,dx2,dx3,dx4 must be from 0.0 to 50.0%% !");
		if ( dy<0 || dy>50 || dy2<0 || dy2>50 || dy3<0 || dy3>50 || dy4<0 || dy4>50 )
			env->ThrowError("DeFreq: dy,dy2,dy3,dy4 must be from 0.0 to 50.0%% !");

	
	if (vi.IsYV12())
	{
		if (plane == 0)
		{ // Y
			nx=vi.width;
			ny=vi.height;
		}
		else if (plane==1 || plane==2) // U,V
		{
			nx=vi.width/2;
			ny=vi.height/2;
		}
		else
			env->ThrowError("DeFreq: plane must be 0,1,2");
	}
	else if (vi.IsYUY2())
	{
		if (plane == 0)
		{ // Y
			nx=vi.width;
			ny=vi.height;
		}
		else if (plane==1 || plane==2) // U,V
		{
			nx=vi.width/2;
			ny=vi.height;
		}
		else
			env->ThrowError("DeFreq: plane must be 0(for Y),1(for U) or 2(for V)");
	}
	else
		env->ThrowError("DeFreq: video must be YV12 or YUY2");


	in = (float *)fftwf_malloc(sizeof(float) * nx*ny);
	outwidth = nx/2+1; // width (pitch) of complex fft
	out = (fftwf_complex *)fftwf_malloc(sizeof(fftwf_complex) * ny*outwidth);
	psd = (float *)malloc(sizeof(float) * ny*outwidth); //v.0.6

	int planMeasure;
	// use FFTW_ESTIMATE or FFTW_MEASURE (more optimal plan, but with time calculation at load stage)
	if (measure && !info && !show) 
		planMeasure = FFTW_MEASURE;
	else 
		planMeasure = FFTW_ESTIMATE;
	plan = fftwf_plan_dft_r2c_2d(ny, nx, in, out, planMeasure); // direct fft 
	plani = fftwf_plan_dft_c2r_2d(ny, nx, out, in, planMeasure); // inverse fft  

	child->SetCacheHints(CACHE_RANGE,0); // range=0 for spatial filter
	
	// init average psd - added in v0.6
	naverage = 0;
	for (int y=0; y<ny; y++) 
	{
		for (int x=0; x<outwidth; x++)
		{
			psd[x] = 0;
		}
		psd += outwidth;
	}
	psd -= outwidth*ny;


}
//-------------------------------------------------------------------------------------------

// This is where any actual destructor code used goes
DeFreq::~DeFreq() {
  // This is where you can deallocate any memory you might have used.
	fftwf_destroy_plan(plan);
	fftwf_destroy_plan(plani);
	fftw_free(in);
	fftw_free(out);	
	free(psd);	
}
//-------------------------------------------------------------------------------------------
//
void DeFreq::SearchPeak(float * psd, int outwidth, int height, float fx, float fy, float dx, float dy, float * fxPeak, float * fyPeak, float *sharpPeak)
{

	// search Peak float percent limits
	float fxmin = max(0, fx - dx);
	float fxmax = min(100, fx + dx);
	float fymin = max(-100, fy - dy);
	float fymax = min(100, fy + dy);

	// from float percent to integer
	int ixmin = int(fxmin * outwidth)/100;
	int ixmax = int(fxmax * outwidth)/100;
	int iymin = int(fymin * height)/200;
	int iymax = int(fymax * height)/200;
	
	float fftmax=-1; // v.0.5
	int hmax = 0;
	int wmax = 0;
	float fftsum = 0;
	int fftcount = 0;
	int h,w;
	for (h=0; h < height/2; h++) 
	{       // Loop from top line to bottom line 
			
			if (h>=iymin && h<=iymax)
			{
				for (w = ixmin; w <= ixmax; w++)       
				{
					float fftcur =	psd[w]; // modul*2
					if (fftcur>fftmax)
					{
						fftmax = fftcur;
						hmax = h;
						wmax = w;
					}
					fftsum += fftcur;
					fftcount++;
				}
			}
			psd += outwidth;
		}

		for (h=height/2; h < height; h++) 
		{       // Loop from top line to bottom line 
			
			if (h >= height+iymin-1 && h <= height+iymax-1)
			{
				for (w = ixmin; w <= ixmax; w++)       
				{
					float fftcur =	psd[w]; // modul*2
					if (fftcur>fftmax)
					{
						fftmax = fftcur;
						hmax = h;
						wmax = w;
					}
					fftsum += fftcur;
					fftcount++;
				}
			}
			psd += outwidth;
	}

	// Peak coordinates (percent)
	*fxPeak = (wmax*100.0f)/outwidth;
	*fyPeak = (hmax < height/2) ? (hmax*200.0f)/height : (hmax - height+1)*200.0f/height;

	float fftbackground;
	if (fftcount>1)
		fftbackground = (fftsum - fftmax)/(fftcount-1);
	else
		fftbackground = 0; // v. 0.5

	//  sharpness
	if (fftbackground > 0) // bux fixed in v.0.5
		*sharpPeak = fftmax/fftbackground;
	else 
		*sharpPeak =1; // bug fixed in v 0.5
}
//-------------------------------------------------------------------------------------------
void DeFreq::CleanPeak(fftwf_complex * outp, int outwidth, int height, float fxPeak, float fyPeak)
{
	// old mode used in Defreq v.0.2, replaced by CleanWindow in v.0.3
	// blind set 12 points to 0
	int w_Peak = int(fxPeak*outwidth)/100;
	int h_Peak = ( fyPeak >=0 ) ? int(fyPeak*height)/200 : int(fyPeak*height)/200 + height-1;// changed to >= in v.0.5

	// must be some array access check here...(later)
			outp += outwidth*h_Peak; // curr line
			outp[w_Peak][0] = 0;
			outp[w_Peak][1] = 0;
			outp[w_Peak+1][0] = 0;
			outp[w_Peak+1][1] = 0;
			if (w_Peak>0)
			{
				outp[w_Peak-1][0] = 0;
				outp[w_Peak-1][1] = 0;
			}
	if (h_Peak-2>=0)// changed to >= in v.0.5
	{
			outp -= outwidth<<1; // prev prev line
			outp[w_Peak][0] = 0;
			outp[w_Peak][1] = 0;
			outp += outwidth<<1; // restore cur line
	}
	if (h_Peak-1>=0)// changed to >= in v.0.5
	{
			outp -= outwidth; // prev line
			outp[w_Peak][0] = 0;
			outp[w_Peak][1] = 0;
			outp[w_Peak+1][0] = 0;
			outp[w_Peak+1][1] = 0;
			if (w_Peak>0)
			{
				outp[w_Peak-1][0] = 0;
				outp[w_Peak-1][1] = 0;
			}
			outp += outwidth; // restore cur line
	}
	if (h_Peak+1<height)
	{
			outp += outwidth; // next line
			outp[w_Peak][0] = 0;
			outp[w_Peak][1] = 0;
			outp[w_Peak+1][0] = 0;
			outp[w_Peak+1][1] = 0;
			if (w_Peak>0)
			{
				outp[w_Peak-1][0] = 0;
				outp[w_Peak-1][1] = 0;
			}
			outp -= outwidth; // restore cur line
	}
	if (h_Peak+2<height)
	{
			outp += outwidth<<1; // next next line
			outp[w_Peak][0] = 0;
			outp[w_Peak][1] = 0;
			outp -= outwidth<<1; // restore cur line
	}
}
//-------------------------------------------------------------------------------------------
//
void DeFreq::CleanWindow(fftwf_complex * out, int outwidth, int height,
						 float fx, float fy, float dx, float dy, float fxPeak, float fyPeak, float sharpPeak)
{
	// more adaptive cleaning method, added in v.0.3
	fftwf_complex * outp = out;

//	float fftnew = fftmax/sharpPeak; // decrease peak to background - no
	float decreasingFactor = sqrtf(sharpPeak); //  factor to clean Peak with neighbors. 
	//I decide to use this empirical value for partial cleaning


	// get Peak value
	int w_Peak = int(fxPeak*outwidth)/100;
	int h_Peak = ( fyPeak >0 ) ? int(fyPeak*height)/200 : int(fyPeak*height)/200 + height-1;
	outp += outwidth*h_Peak; // curr line
	float fftmax = (outp[w_Peak][0])*(outp[w_Peak][0])+(outp[w_Peak][1])*(outp[w_Peak][1]);


	outp = out; // pointer

	// search Peak limits
	float fxmin = max(0, fx - dx);
	float fxmax = min(100, fx + dx);
	float fymin = max(-100, fy - dy);
	float fymax = min(100, fy + dy);
	
	// from float percent to integer
	int ixmin = int(fxmin * outwidth)/100;
	int ixmax = int(fxmax * outwidth)/100;
	int iymin = int(fymin * height)/200;
	int iymax = int(fymax * height)/200;

	int hmax = 0;
	int wmax = 0;
	float fftsum = 0;
	int fftcount = 0;
	int h,w;
	for (h=0; h < height/2; h++) 
	{       // Loop from top line to middle line 
			
			if (h>=iymin && h<=iymax)
			{
				for (w = ixmin; w <= ixmax; w++)       
				{
					float fftcur =	(outp[w][0])*(outp[w][0])+(outp[w][1])*(outp[w][1]); // modul*2
					if (fftcur*decreasingFactor > fftmax) // use decreasingFactor also as measure of similarity to peak
					{
						outp[w][0] /= decreasingFactor; // decrease
						outp[w][1] /= decreasingFactor; // decrease

					}
				}
			}
			outp += outwidth;
		}

		for (h=height/2; h < height; h++) 
		{       // Loop from middle line to bottom line 
			
			if (h >= height+iymin-1 && h <= height+iymax-1)
			{
				for (w = ixmin; w <= ixmax; w++)       
				{
					float fftcur =	(outp[w][0])*(outp[w][0])+(outp[w][1])*(outp[w][1]); // modul*2
					if (fftcur*decreasingFactor > fftmax)
					{
						outp[w][0] /= decreasingFactor;
						outp[w][1] /= decreasingFactor;
					}
				}
			}
			outp += outwidth;
	}

}
//-------------------------------------------------------------------------------------------
void DeFreq::CleanHigh(fftwf_complex * outp, int outwidth, int height, float cutx, float cuty)
{
	// Butterworth low pass filter, 
	// factor = 1/( (freqx/cutx)*(freqx/cutx) + (freqy/cuty)*(freqy/cuty)
	int w,h;
	float fh,fw,f;
	float invcutx = 100.0f/(cutx*outwidth);
	float invcuty = 200.0f/(cuty*height);
	for (h=0; h < height/2; h++) 
	{      
			
			{
				fh = float(h)*invcuty;
				fh *= fh;
				for (w = 0; w < outwidth; w++)       
				{
					fw = float(w)*invcutx;
					fw *= fw;
					f = 1/(1 + fh + fw);
					outp[w][0] *= f;
					outp[w][1] *= f;
				}
			}
			outp += outwidth;
		}

		for (h=height/2; h < height; h++) 
		{    
			
			{
				fh = float(height-h-1)*invcuty;
				fh *= fh;
				for (w = 0; w < outwidth; w++)       
				{
					fw = float(w)*invcutx;
					fw *= fw;
					f = 1/(1 + fh + fw);
					outp[w][0] *= f;
					outp[w][1] *= f;
				}
			}
			outp += outwidth;
	}
}
//-------------------------------------------------------------------------------------------
void DeFreq::GetFFT2minmax(float * psd, int outwidth, int height, float *fft2min, float * fft2max) // changed in v0.6
{
		// found fftmin and fftmax
		float fft2cur;
		float psdmin = 1.0e14f; // bug fixed in v.0.4a
		float psdmax = 0;
		for (int h=0; h < height; h++) 
		{   
			for (int w = 0; w < outwidth; w++) 
			{
				fft2cur = psd[w]; 
//				if (fft2cur < psdmin) 
//				{
//					psdmin = fft2cur;
//				}
				if (fft2cur > psdmax) 
				{
					psdmax = fft2cur;
				}
			}
			psd += outwidth;
		}
		if (psdmax == 0)
		{
			psdmax = 1.0f;
		}
		psdmin = (psdmax)*1.0e-13f; // some empirical number - v0.6
		*fft2min = psdmin;
		*fft2max = psdmax;
}
//-------------------------------------------------------------------------------------------
void DeFreq::DrawSearchBox(float * psd, int outwidth, int height, float fx, float fy, float dx, float dy, float fftvalue)
{
	float fxmin = max(0, fx - dx);
	float fxmax = min(100, fx + dx);
	float fymin = max(-100, fy - dy);
	float fymax = min(100, fy + dy);

	// from float percent to integer
	int ixmin = int(fxmin * outwidth)/100;
	int ixmax = int(fxmax * outwidth)/100;
	int iymin = int(fymin * height)/200;
	int iymax = int(fymax * height)/200;

	int hmin = ( iymin >=0 ) ? iymin : iymin + height-1; // changed to >= in v.0.5
	int hmax = ( iymax >=0 ) ? iymax : iymax + height-1; // changed to >= in v.0.5
	int w,h;
	if (hmin <= hmax) // one region
	{
		psd += outwidth*hmin;
		for (w=ixmin; w<=ixmax; w++)
		{
			psd[w] = fftvalue;
		}
		for (h=hmin; h<hmax; h++)
		{
			psd[ixmin] = fftvalue;
			psd[ixmax] = fftvalue;
			psd += outwidth;
		}
		for (w=ixmin; w<=ixmax; w++)
		{
			psd[w] = fftvalue;
		}
	}
	else // two regions (fymin<0, fymax>0    i.e.      hmin>height/2, hmax<height/2)
	{
//		for (w=ixmin; w<ixmax; w++) // disabled in v0.6
		{
//			psd[w] = fftvalue;
		}
		for (h=0; h<hmax; h++)
		{
			psd[ixmin] = fftvalue;
			psd[ixmax] = fftvalue;
			psd += outwidth;
		}
		for (w=ixmin; w<ixmax; w++)
		{
			psd[w] = fftvalue;
		}
		psd += outwidth*(hmin-hmax);
		for (w=ixmin; w<ixmax; w++)
		{
			psd[w] = fftvalue;
		}
		for (h=hmin; h<height; h++)
		{
			psd[ixmin] = fftvalue;
			psd[ixmax] = fftvalue;
			psd += outwidth;
		}
//		for (w=ixmin; w<ixmax; w++) // disabled in v0.6
		{
//			psd[w] = fftvalue;
		}
	}

// must be added case fymax<fymin

}
//-------------------------------------------------------------------------------------------
void DeFreq::FrequencySwitchOn(fftwf_complex * outp, int outwidth, int height, float fx, float fy, float setvalue)
{
	// from float percent to integer
	int ix = int(fx * outwidth)/100;
	int iy = int(fy * height)/200;
	
	int h = ( iy >= 0 ) ? iy : iy + height-1;
	outp += outwidth*h;
	outp[ix][0] = setvalue;
//	outp[ix][1] = setvalue;


}//-------------------------------------------------------------------------------------------
//
// main code for YV12
//
void DeFreq::DeFreqYV12(BYTE *srcp0, int src_height, int src_width, int src_pitch,
						float * fxPeak, float * fyPeak, float *sharpPeak,
						float * fxPeak2, float * fyPeak2, float *sharpPeak2,
						float * fxPeak3, float * fyPeak3, float *sharpPeak3,
						float * fxPeak4, float * fyPeak4, float *sharpPeak4)
{
	float * inp = in; 
	fftwf_complex * outp = out; //pointer
	
	int w, h;
	int width_2 = src_width/2; // half

	BYTE * srcp = srcp0;
	BYTE * dstp = srcp0;
	
		
	// byte to float
	for (h=0; h < src_height; h++) 
	{       // Loop from top line to bottom line 
		for (w = 0; w < src_width; w++)       // Loop from left side of the image to the right side.
		{
			inp[w] = float(srcp[w]);          // Copy each byte from source to float array
			// may be ASM optimized later (from Vaguedenoiser code ?)
		}
		srcp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
		inp += nx; // pitch of float array
	}
	inp -= nx*src_height;
	srcp -= src_pitch*src_height;

	fftwf_execute_dft_r2c(plan, inp, out); // do FFT, get out

	if (show==2) // v0.6
		naverage += 1; // average
	else
		naverage = 1; // no average
	float faverage = 1.0f/naverage;
	for (h=0; h<src_height; h++)
	{
		for (w=0; w<outwidth; w++)
		{// average power spectral density
			psd[w] = psd[w]*(1-faverage) + (outp[w][0]*outp[w][0] + outp[w][1]*outp[w][1])*faverage; 
		}
		psd += outwidth;
		outp += outwidth;
	}
	psd -= outwidth*src_height;
	outp -= outwidth*src_height;
	
	

	// search Peaks
	if (fx>0 || fy !=0) // changed from AND to OR (to allow one of fx or fy be =0) in v.0.5
		DeFreq::SearchPeak(psd, outwidth, src_height, fx, fy, dx, dy, fxPeak, fyPeak, sharpPeak);
	if (fx2>0 || fy2 !=0)
		DeFreq::SearchPeak(psd, outwidth, src_height, fx2, fy2, dx2, dy2, fxPeak2, fyPeak2, sharpPeak2);
	if (fx3>0 || fy3 !=0)
		DeFreq::SearchPeak(psd, outwidth, src_height, fx3, fy3, dx3, dy3, fxPeak3, fyPeak3, sharpPeak3);
	if (fx4>0 || fy4 !=0)
		DeFreq::SearchPeak(psd, outwidth, src_height, fx4, fy4, dx4, dy4, fxPeak4, fyPeak4, sharpPeak4);


	if (show) 
	{	// mode -  show fft 
		// found fftmin and fftmax
		float fft2min = 0;
		float fft2max = 0;
		DeFreq::GetFFT2minmax(psd, outwidth, src_height, &fft2min, &fft2max);

		// draw search windows
		if (fx>0 || fy !=0)// changed from AND to OR (to allow one of fx or fy be =0) in v.0.5
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx, fy, dx, dy, fft2max);
		if (fx2>0 || fy2 !=0)
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx2, fy2, dx2, dy2, fft2max);
		if (fx3>0 || fy3 !=0)
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx3, fy3, dx3, dy3, fft2max);
		if (fx4>0 || fy4 !=0)
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx4, fy4, dx4, dy4, fft2max);

		float logmin = logf(fft2min);
		float logmax = logf(fft2max);
		float fac = 255.5f/(logmax - logmin);
		// show fft logarifm of power density on left half of output image
//		outp = out;
/*		for (h=0; h < src_height; h++) 
			{       
				for (w = 0; w < width_2; w++)       // left half
				{
					int dstcur = (int)( fac*(logf(psd[w] + 1e-15) - logmin ));
					dstp[w] = min(255,max(0,dstcur));  
				}
				dstp += src_pitch;            // Add the pitch 
				psd += outwidth;
		}
		dstp -= src_pitch*src_height;
		psd -= outwidth*src_height;
*/
		// revert and stack spectrum subwindows - view changed in v0.6
		psd += (src_height/2)*outwidth; // middle positive
		for (h=0; h < src_height/2; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_2; w++)       // left half
				{
					int dstcur = (int)( fac*(log(psd[w] + 1e-15) - logmin ));
					dstp[w] = min(255,max(0,dstcur));  
				}
				dstp += src_pitch;            // Add the pitch 
		}
		psd += (src_height)*outwidth; // bottom negative
		for (h=src_height/2; h < src_height; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_2; w++)       // left half
				{
					int dstcur = (int)( fac*(log(psd[w] + 1e-15) - logmin ));
					dstp[w] = min(255,max(0,dstcur));  
				}
				dstp += src_pitch;            // Add the pitch 
		}
		dstp -= src_pitch*src_height;
		psd -= outwidth*(src_height -(src_height/2));
		
		// code to show test frequencies stripes on right top quarter of output image - added in v.0.5
		
		// clear spectrum
//		outp = out; // pointer
		for (h=0; h < src_height; h++) 
			{       
				for (w = 0; w < outwidth; w++)     
				{
					outp[w][0] = 0;
					outp[w][1] = 0;
				}
				outp += outwidth;
		}
		outp -= outwidth*src_height;

		outp[0][0] = 128;// set general brightness = 128

		// count of search windows
		float weight = 0;
		if (fx>0 || fy !=0)
			weight += 1.0f;;
		if (fx2>0 || fy2 !=0)
			weight += 1/1.4f;
		if (fx3>0 || fy3 !=0)
			weight += 1/2.0f;
		if (fx4>0 || fy4 !=0)
			weight += 1/2.8f;
		// set frequency amplitude to some high value at central search frequencies
		float setvalue = 60.0f/(weight + 0.0001f);
		if (fx>0 || fy !=0)
			DeFreq::FrequencySwitchOn(outp, outwidth, src_height, fx, fy, setvalue);
		if (fx2>0 || fy2 !=0)
			DeFreq::FrequencySwitchOn(outp, outwidth, src_height, fx2, fy2, setvalue/1.4f); // decrease a litle (1.4) 
		if (fx3>0 || fy3 !=0)
			DeFreq::FrequencySwitchOn(outp, outwidth, src_height, fx3, fy3, setvalue/2.0f);// decrease a litle (1.4) 
		if (fx4>0 || fy4 !=0)
			DeFreq::FrequencySwitchOn(outp, outwidth, src_height, fx4, fy4, setvalue/2.8f);// decrease a litle (1.4) 


		// do inverse fft
		fftwf_execute_dft_c2r(plani, outp, in); // iFFT

		// show test frequencies stripes on right top quarter of output image
			
		// float to byte
		inp = in; // pointer
		dstp = srcp0;
		for (h=0; h < src_height/2; h++) 
		{   // Loop from top line to bottom line 
			for (w = width_2; w < src_width; w++)  // right half 
			{
				int result =(int)(inp[w]); // normalize FFT result
				dstp[w] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
				// may be ASM optimized later (from Vaguedenoiser code ?)
			}
			dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
			inp += nx; // pitch of float array
		}
		dstp -= src_pitch*(src_height/2);
		inp -= nx*(src_height/2);
	}
	else
	{ //  mode - work

		bool clean = false; // reset clean flag for needed processing (will be set to true if must be cleaned) - added in 0.5.1

		// check sharpness and clean Peak frequency with some neighbors in window
		if ( *sharpPeak > sharp)
		{
//			DeFreq::CleanPeak(out, outwidth, src_height, *fxPeak, *fyPeak); // replaced in v.0.3
			DeFreq::CleanWindow(out, outwidth, src_height, fx, fy, dx, dy, *fxPeak, *fyPeak, *sharpPeak);
			clean = true; //- added in 0.5.1
		}
		if ( *sharpPeak2 > sharp2) 
		{
			DeFreq::CleanWindow(out, outwidth, src_height, fx2, fy2, dx2, dy2, *fxPeak2, *fyPeak2, *sharpPeak2);
			clean = true;
		}
		if ( *sharpPeak3 > sharp3)	
		{
			DeFreq::CleanWindow(out, outwidth, src_height, fx3, fy3, dx3, dy3, *fxPeak3, *fyPeak3, *sharpPeak3);
			clean = true;
		}
		if ( *sharpPeak4 > sharp4) 
		{
			DeFreq::CleanWindow(out, outwidth, src_height, fx4, fy4, dx4, dy4, *fxPeak4, *fyPeak4, *sharpPeak4);
			clean = true;
		}

		if (cutx>0 && cuty>0 ) 		// clean high frequencies
		{
			DeFreq::CleanHigh(out, outwidth, src_height, cutx, cuty);
			clean = true;
		}

		if (clean) // some frequencies must be cleaned ?  (if not, skip further unneeded processing)- added in 0.5.1
		{
			// do inverse fft
			fftwf_execute(plani); // iFFT

			// make output frame plane
				
			// float to byte
			
			float norm = 1.0f/(nx*ny); // FFT normalization factor
			inp = in; // pointer
			for (h=0; h < src_height; h++) 
			{   // Loop from top line to bottom line 
				for (w = 0; w < src_width; w++)   // Loop from left side of the image to the right side.
				{
					int result =(int)(inp[w]*norm); // normalize FFT result
					dstp[w] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
					// may be ASM optimized later (from Vaguedenoiser code ?)
				}
				dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
				inp += nx; // pitch of float array
			}
		}
	}

}
//-------------------------------------------------------------------------------------------

//
// main code for YUY2
//
void DeFreq::DeFreqYUY2(int plane, BYTE *srcp0, int src_height, int src_width, int src_pitch,
						float * fxPeak, float * fyPeak, float *sharpPeak,
						float * fxPeak2, float * fyPeak2, float *sharpPeak2,
						float * fxPeak3, float * fyPeak3, float *sharpPeak3,
						float * fxPeak4, float * fyPeak4, float *sharpPeak4)
{
	float * inp = in; 
	
	int w, h;
	int width_2 = src_width/2; // half
	int width_4 = src_width/4; 
	int width_8 = src_width/8; 

	BYTE * srcp = srcp0;
	BYTE * dstp = srcp0;
	fftwf_complex * outp = out;
	
	// byte to float
		// This code deals with YUY2 colourspace where each 4 byte sequence represents
		// 2 pixels, (Y1, U, Y2 and then V).
	if (plane==0)
	{
		for (h=0; h < src_height; h++) 
		{       // Loop from top line to bottom line 
			for (w = 0; w < width_2; w++)       // Loop from left side of the image to the right side.
			{
				inp[w] = float(srcp[w<<1]);          // Copy each byte from source to float array
			}
			srcp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
			inp += nx; // pitch of float array
		}
		inp -= nx*src_height;
		srcp -= src_pitch*src_height;
	}
	else if (plane==1) //U
	{
		for (h=0; h < src_height; h++) 
		{       // Loop from top line to bottom line 
			for (w = 0; w < width_4; w++)       
			{
				inp[w] = float(srcp[(w<<2)+1]);          // Copy each byte from source to float array
			}
			srcp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
			inp += nx; // pitch of float array
		}
		inp -= nx*src_height;
		srcp -= src_pitch*src_height;
	}
	else if (plane==2) //V
	{
		for (h=0; h < src_height; h++) 
		{      
			for (w = 0; w < width_4; w++)       
			{
				inp[w] = float(srcp[(w<<2)+3]);    // Copy each byte from source to float array
			}
			srcp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
			inp += nx; // pitch of float array
		}
		inp -= nx*src_height;
		srcp -= src_pitch*src_height;
	}

	fftwf_execute_dft_r2c(plan, in, out); // do FFT

	if (show==2) 
		naverage += 1; // average
	else
		naverage = 1; // no average

	float faverage = 1.0f/naverage;
	
	for (h=0; h<ny; h++)
	{
		for (w=0; w<outwidth; w++)
		{// average power spectral density
			psd[w] = psd[w]*(1-faverage) + (outp[w][0]*outp[w][0] + outp[w][1]*outp[w][1])*faverage;
		}
		psd += outwidth;
		outp += outwidth;
	}
	psd -= outwidth*ny;
	outp -= outwidth*ny;
	
	
//	outp = out;

	// search Peak in windows
	if (fx>0 || fy !=0)// changed from AND to OR (to allow one of fx or fy be =0) in v.0.5
		DeFreq::SearchPeak(psd, outwidth, src_height, fx, fy, dx, dy, fxPeak, fyPeak, sharpPeak);
	if (fx2>0 || fy2 !=0)
		DeFreq::SearchPeak(psd, outwidth, src_height, fx2, fy2, dx2, dy2, fxPeak2, fyPeak2, sharpPeak2); // bug fixed in v.0.5.4
	if (fx3>0 || fy3 !=0)
		DeFreq::SearchPeak(psd, outwidth, src_height, fx3, fy3, dx3, dy3, fxPeak3, fyPeak3, sharpPeak3);
	if (fx4>0 || fy4 !=0)
		DeFreq::SearchPeak(psd, outwidth, src_height, fx4, fy4, dx4, dy4, fxPeak4, fyPeak4, sharpPeak4);



	if (show) 
	{	// mode -  show fft 
		float fft2min = 0;
		float fft2max = 0;
		DeFreq::GetFFT2minmax(psd, outwidth, src_height, &fft2min, &fft2max);
		// Draw search windows
		if (fx>0 || fy !=0)// changed from AND to OR (to allow one of fx or fy be =0) in v.0.5
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx, fy, dx, dy, fft2max);
		if (fx2>0 || fy2 !=0)
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx2, fy2, dx2, dy2, fft2max);
		if (fx3>0 || fy3 !=0)
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx3, fy3, dx3, dy3, fft2max);
		if (fx4>0 || fy4 !=0)
			DeFreq::DrawSearchBox(psd, outwidth, src_height, fx4, fy4, dx4, dy4, fft2max);

		float logmin = logf(fft2min);
		float logmax = logf(fft2max);
		float fac = 255.5f/(logmax - logmin);
		// show fft power density (log)
		outp = out;
		dstp = srcp0;
		if (plane==0)
		{
			psd += (src_height/2)*outwidth; // middle positive
			for (h=0; h < src_height/2; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_4; w+=2)       // Loop from left side of the image to the right side.
				{
					int dstcur = (int)( fac*(log(psd[w] + 1e-15) - logmin ));
					dstp[w<<1] = min(255,max(0,dstcur));  
					dstp[(w<<1)+1] = 128;
					dstcur = (int)( fac*(log(psd[w+1] + 1e-15) - logmin )); // bug fixed in v.0.5
					dstp[(w<<1)+2] = min(255,max(0,dstcur));  
					dstp[(w<<1)+3] = 128;
				}
				dstp += src_pitch;            // Add the pitch 
			}
			psd += (src_height)*outwidth; // bottom negative
			for (h=src_height/2; h < src_height; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_4; w+=2)       // Loop from left side of the image to the right side.
				{
					int dstcur = (int)( fac*(log(psd[w] + 1e-15) - logmin ));
					dstp[w<<1] = min(255,max(0,dstcur));  
					dstp[(w<<1)+1] = 128;
					dstcur = (int)( fac*(log(psd[w+1]+1e-15) - logmin )); // bug fixed in v.0.5
					dstp[(w<<1)+2] = min(255,max(0,dstcur));  
					dstp[(w<<1)+3] = 128;
				}
				dstp += src_pitch;            // Add the pitch 
			}
			dstp -= src_pitch*src_height;
			psd -= outwidth*(src_height -(src_height/2));
		}
		else if (plane==1) //U
		{
			psd += (src_height/2)*outwidth; // middle positive
			for (h=0; h < src_height/2; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_8; w++)       // Loop from left side of the image to the right side.
				{
					dstp[w<<2] = 128;
					int dstcur = (int)( fac*(log(psd[w]+1e-15 ) - logmin ));
					dstp[(w<<2)+1] = min(255,max(0,dstcur));  
					dstp[(w<<2)+2] = 128;
					dstp[(w<<2)+3] = 128;
				}
				dstp += src_pitch;            // Add the pitch 
			}
			psd += (src_height)*outwidth; // bottom negative
			for (h=src_height/2; h < src_height; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_8; w++)       // Loop from left side of the image to the right side.
				{
					dstp[w<<2] = 128;
					int dstcur = (int)( fac*(log(psd[w] +1e-15) - logmin) );
					dstp[(w<<2)+1] = min(255,max(0,dstcur));  
					dstp[(w<<2)+2] = 128;
					dstp[(w<<2)+3] = 128;
				}
				dstp += src_pitch;            // Add the pitch 
			}
			dstp -= src_pitch*src_height;
			psd -= outwidth*(src_height -(src_height/2));
		}
		else if (plane==2) //V
		{
			psd += (src_height/2)*outwidth; // middle positive
			for (h=0; h < src_height/2; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_8; w++)       // Loop from left side of the image to the right side.
				{
					dstp[w<<2] = 128;
					dstp[(w<<2)+1] = 128;
					dstp[(w<<2)+2] = 128;
					int dstcur = (int)( fac*(log(psd[w] + 1e-15) - logmin ));
					dstp[(w<<2)+3] = min(255,max(0,dstcur));  
				}
				dstp += src_pitch;            // Add the pitch 
			}
			psd += (src_height)*outwidth; // bottom negative
			for (h=src_height/2; h < src_height; h++) 
			{       
				psd -= outwidth;
				for (w = 0; w < width_8; w++)       // Loop from left side of the image to the right side.
				{
					dstp[w<<2] = 128;
					dstp[(w<<2)+1] = 128;
					dstp[(w<<2)+2] = 128;
					int dstcur = (int)( fac*(log(psd[w] + 1e-15) - logmin ));
					dstp[(w<<2)+3] = min(255,max(0,dstcur));  
				}
				dstp += src_pitch;            // Add the pitch 
			}
			dstp -= src_height*src_pitch;
			psd -= outwidth*(src_height -(src_height/2));
		}

		// code to show test frequencies stripes on right top quarter of output image - added in v.0.5
		
		// clear spectrum
		outp = out; // pointer
		for (h=0; h < src_height; h++) 
			{       
				for (w = 0; w < outwidth; w++)     
				{
					outp[w][0] = 0;
					outp[w][1] = 0;
				}
				outp += outwidth;
		}
		outp -= src_height*outwidth;

		out[0][0] = 128;// set general brightness = 128

		// count of search windows
		float weight = 0;
		if (fx>0 || fy !=0)
			weight += 1.0f;;
		if (fx2>0 || fy2 !=0)
			weight += 1/1.4f;
		if (fx3>0 || fy3 !=0)
			weight += 1/2.0f;
		if (fx4>0 || fy4 !=0)
			weight += 1/2.8f;
		// set frequency amplitude to some high value at central search frequencies
		float setvalue = 60.0f/(weight + 0.0001f);
		if (fx>0 || fy !=0)
			DeFreq::FrequencySwitchOn(out, outwidth, src_height, fx, fy, setvalue);
		if (fx2>0 || fy2 !=0)
			DeFreq::FrequencySwitchOn(out, outwidth, src_height, fx2, fy2, setvalue/1.4f); // decrease a litle (1.4) 
		if (fx3>0 || fy3 !=0)
			DeFreq::FrequencySwitchOn(out, outwidth, src_height, fx3, fy3, setvalue/2.0f);// decrease a litle (1.4) 
		if (fx4>0 || fy4 !=0)
			DeFreq::FrequencySwitchOn(out, outwidth, src_height, fx4, fy4, setvalue/2.8f);// decrease a litle (1.4) 


		// do inverse fft
		fftwf_execute_dft_c2r(plani, out, in); // iFFT

		// show test frequencies stripes on right top quarter of output image
			
		// float to byte
		
		inp = in; // pointer
		dstp = srcp0;
		if (plane==0)
		{
			for (h=0; h < src_height/2; h++) 
			{   // Loop from top line to bottom line 
				for (w = width_4; w < width_2; w++)   // to the right side.
				{
					int result = (int)(inp[w]); 
					dstp[w<<1] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
					dstp[(w<<1)+1] = 128; // grey
					// may be ASM optimized later (from Vaguedenoiser code ?)
				}
				dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
				inp += nx; // pitch of float array
			}
		}
		else if (plane==1) //U
		{
			for (h=0; h < src_height/2; h++) 
			{   // Loop from top line to bottom line 
				for (w = width_8; w < width_4; w++)   // to the right side.
				{
					int result = (int)(inp[w]); 
					dstp[(w<<2)] =0; // Y
					dstp[(w<<2)+1] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
					dstp[(w<<2)+2] =0; // Y
					dstp[(w<<2)+3] = 128; // V grey
					// may be ASM optimized later (from Vaguedenoiser code ?)
				}
				dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
				inp += nx; // pitch of float array
			}
		}
		else if (plane==2) //V
		{
			for (h=0; h < src_height/2; h++) 
			{   // Loop from top line to bottom line 
				for (w = width_8; w < width_4; w++)   //  to the right side.
				{
					int result = (int)(inp[w]); 
					dstp[(w<<2)] =0;// Y
					dstp[(w<<2)+1] = 128; // U grey
					dstp[(w<<2)+2] = 0;// Y //smalll bux fixed in v0.6
					dstp[(w<<2)+3] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
					// may be ASM optimized later (from Vaguedenoiser code ?)
				}
				dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
				inp += nx; // pitch of float array
			}
		}

	}
	else
	{ // mode - work ------------------------------

		bool clean = false; // reset clean flag (will be set to true if must be cleaned) - added in 0.5.1

		// check sharpness and clean Peak with neighbors
		if ( *sharpPeak > sharp)
		{
//			DeFreq::CleanPeak(out, outwidth, src_height, *fxPeak, *fyPeak); // replaced in v.0.3
			DeFreq::CleanWindow(out, outwidth, src_height, fx, fy, dx, dy,  *fxPeak, *fyPeak, *sharpPeak);
			clean = true; // - added in 0.5.1
		}
		if ( *sharpPeak2 > sharp2) 
		{
			DeFreq::CleanWindow(out, outwidth, src_height, fx2, fy2, dx2, dy2,  *fxPeak2, *fyPeak2, *sharpPeak2);
			clean = true;
		}
		if ( *sharpPeak3 > sharp3) 
		{
			DeFreq::CleanWindow(out, outwidth, src_height, fx3, fy3, dx3, dy3, *fxPeak3, *fyPeak3, *sharpPeak3);
			clean = true;
		}
		if ( *sharpPeak4 > sharp4)
		{
			DeFreq::CleanWindow(out, outwidth, src_height, fx4, fy4, dx4, dy4, *fxPeak4, *fyPeak4, *sharpPeak4);
			clean = true;
		}

		if (cutx>0 && cuty>0 ) 		// clean high frequencies
		{
			DeFreq::CleanHigh(out, outwidth, src_height, cutx, cuty);
			clean = true;
		}

		if (clean) // must be clean for some reason?  (if not, do nothing) - added in 0.5.1
		{
			// do inverse fft
			fftwf_execute(plani); // iFFT

			// make output frame

			// float to byte
			
			float norm = 1.0f/(nx*ny); // FFT normalization factor
			inp = in; // pointer
			if (plane==0)
			{
				for (h=0; h < src_height; h++) 
				{   // Loop from top line to bottom line 
					for (w = 0; w < width_2; w++)   // Loop from left side of the image to the right side.
					{
						int result = (int)(inp[w]*norm); // normalize FFT result
						dstp[w<<1] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
						// may be ASM optimized later (from Vaguedenoiser code ?)
					}
					dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
					inp += nx; // pitch of float array
				}
			}
			else if (plane==1) //U
			{
				for (h=0; h < src_height; h++) 
				{   // Loop from top line to bottom line 
					for (w = 0; w < width_4; w++)   // Loop from left side of the image to the right side.
					{
						int result = (int)(inp[w]*norm); // normalize FFT result
						dstp[(w<<2)+1] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
						// may be ASM optimized later (from Vaguedenoiser code ?)
					}
					dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
					inp += nx; // pitch of float array
				}
			}
			else if (plane==2) //V
			{
				for (h=0; h < src_height; h++) 
				{   // Loop from top line to bottom line 
					for (w = 0; w < width_4; w++)   // Loop from left side of the image to the right side.
					{
						int result = (int)(inp[w]*norm); // normalize FFT result
						dstp[(w<<2)+3] = min(255,max(0,result)); // Put each  float result (clipped) to byte dest 
						// may be ASM optimized later (from Vaguedenoiser code ?)
					}
					dstp += src_pitch;            // Add the pitch of one line (in bytes) to the source image.
					inp += nx; // pitch of float array
				}
			}
		}
	}
}

//-------------------------------------------------------------------------------------------
// set all plane bytes to same value (grey)
// MMX version would be faster - but it is used only in show mode 
void mem_set(BYTE *dest, int value, int height, int width,  int pitch)
{
	for (int h=0; h<height; h++)
	{
		for (int w=0; w<width; w++)
		{
			dest[w] = value;
		}
		dest += pitch;
	}
}

//-------------------------------------------------------------------------------------------

PVideoFrame __stdcall DeFreq::GetFrame(int n, IScriptEnvironment* env) {
// This is the implementation of the GetFrame function.
// See the header definition for further info.

	// init Peaks to be found
	float fxPeak=0; //  fixed bug with init in v.0.4.1
	float fyPeak=0;
	float sharpPeak=0; 
	float fxPeak2=0; 
	float fyPeak2=0;
	float sharpPeak2=0;
	float fxPeak3=0; 
	float fyPeak3=0;
	float sharpPeak3=0;
	float fxPeak4=0; 
	float fyPeak4=0;
	float sharpPeak4=0;

	bool IsYUY2 = vi.IsYUY2(); // added in v 0.5

	PVideoFrame src = child->GetFrame(n, env);
   // Request frame 'n' from the child (source) clip.

 //If you recieve a frame from PClip->GetFrame(...)
  // you must call env->MakeWritable(&frame) be recieve a valid write pointer.
	env->MakeWritable(&src);

	BYTE * srcp;
	int src_pitch;
	int src_width;
	int src_height;

	if (vi.IsYV12())
	{
		// This code deals with YV12 colourspace where the Y, U and V information are
		// stored in completely separate memory areas

		if (show)
		{ // in show mode, set other planes to 128

			if (plane==0) // work plane Y
			{ 
				// U
				srcp = src->GetWritePtr(PLANAR_U);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_U);
				src_width = src->GetRowSize(PLANAR_U);
				src_height = src->GetHeight(PLANAR_U);
//				memset(srcp, 128, src_pitch*src_height);// set to 128 - it has some bug
				mem_set(srcp, 128,  src_height, src_width, src_pitch); // set to 128,  bug fixed in v.0.5.3
				// V
				srcp = src->GetWritePtr(PLANAR_V);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_V);
				src_width = src->GetRowSize(PLANAR_V);
				src_height = src->GetHeight(PLANAR_V);
				mem_set(srcp, 128,  src_height, src_width, src_pitch); // set to 128
				// Y
				srcp = src->GetWritePtr(PLANAR_Y);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_Y);
				src_width = src->GetRowSize(PLANAR_Y);
				src_height = src->GetHeight(PLANAR_Y); 
			}
			else if(plane==1) // work plane U
			{
				// Y
				srcp = src->GetWritePtr(PLANAR_Y);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_Y);
				src_width = src->GetRowSize(PLANAR_Y);
				src_height = src->GetHeight(PLANAR_Y); 
				mem_set(srcp, 128,  src_height, src_width, src_pitch); // set to 128
				// V
				srcp = src->GetWritePtr(PLANAR_V);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_V);
				src_width = src->GetRowSize(PLANAR_V);
				src_height = src->GetHeight(PLANAR_V);
				mem_set(srcp, 128,  src_height, src_width, src_pitch); // set to 128
				// U
				srcp = src->GetWritePtr(PLANAR_U);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_U);
				src_width = src->GetRowSize(PLANAR_U);
				src_height = src->GetHeight(PLANAR_U);
			}
			else if(plane==2) // work plane V
			{
				// Y
				srcp = src->GetWritePtr(PLANAR_Y);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_Y);
				src_width = src->GetRowSize(PLANAR_Y);
				src_height = src->GetHeight(PLANAR_Y); 
				mem_set(srcp, 128,  src_height, src_width, src_pitch); // set to 128
				// U
				srcp = src->GetWritePtr(PLANAR_U);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_U);
				src_width = src->GetRowSize(PLANAR_U);
				src_height = src->GetHeight(PLANAR_U);
				mem_set(srcp, 128,  src_height, src_width, src_pitch); // set to 128
				// V
				srcp = src->GetWritePtr(PLANAR_V);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_V);
				src_width = src->GetRowSize(PLANAR_V);
				src_height = src->GetHeight(PLANAR_V);
			}

		}
		else // clean mode
		{
			if (plane==0) // Y
			{ 
				srcp = src->GetWritePtr(PLANAR_Y);  // Request a  pointer from the source frame.
				  src_pitch = src->GetPitch(PLANAR_Y);
				  src_width = src->GetRowSize(PLANAR_Y);
				  src_height = src->GetHeight(PLANAR_Y);
			}
			else if(plane==1) // U
			{
				srcp = src->GetWritePtr(PLANAR_U);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_U);
				src_width = src->GetRowSize(PLANAR_U);
				src_height = src->GetHeight(PLANAR_U);
			}
			else if(plane==2) // V
			{
				srcp = src->GetWritePtr(PLANAR_V);  // Request a  pointer from the source frame.
				src_pitch = src->GetPitch(PLANAR_V);
				src_width = src->GetRowSize(PLANAR_V);
				src_height = src->GetHeight(PLANAR_V);
			}
		}
		DeFreq::DeFreqYV12(srcp,src_height,src_width,src_pitch, &fxPeak, &fyPeak, &sharpPeak,
			&fxPeak2, &fyPeak2,  &sharpPeak2, &fxPeak3, &fyPeak3,  &sharpPeak3, &fxPeak4, &fyPeak4,  &sharpPeak4);

		if (info)
		{	// show frequency info
				int x0 = vi.width/20+1;
				int y0 = 0;
				DrawString(src, x0, y0++, "DeFreq peaks:",IsYUY2); // YUY2 compatible version of info.h added in v.0.5
				if (fx >0 || fy !=0) // changed to OR in v.0.5
				{ //first Peak
					sprintf(messagebuf,"x=%.1f y=%.1f", fxPeak, fyPeak);
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak > sharp)
						sprintf(messagebuf,"SHARP=%.1f", sharpPeak);
					else
						sprintf(messagebuf,"sharp=%.1f", sharpPeak);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
				if (fx2 >0 || fy2 !=0)
				{ //second Peak
					sprintf(messagebuf,"x2=%.1f y2=%.1f", fxPeak2, fyPeak2);
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak2 > sharp2)
						sprintf(messagebuf,"SHARP2=%.1f", sharpPeak2);
					else
						sprintf(messagebuf,"sharp2=%.1f", sharpPeak2);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
				if (fx3 >0 || fy3 !=0)
				{ // third Peak
					sprintf(messagebuf,"x3=%.1f y3=%.1f", fxPeak3, fyPeak3);
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak3 > sharp3)
						sprintf(messagebuf,"SHARP3=%.1f", sharpPeak3);
					else
						sprintf(messagebuf,"sharp3=%.1f", sharpPeak3);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
				if (fx4 >0 || fy4 !=0)
				{ // fourth Peak
					sprintf(messagebuf,"x4=%.1f y4=%.1f", fxPeak4, fyPeak4);
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak4 > sharp4)
						sprintf(messagebuf,"SHARP4=%.1f", sharpPeak4);
					else
						sprintf(messagebuf,"sharp4=%.1f", sharpPeak4);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
		}
	}
	else if (vi.IsYUY2())
	{
		srcp = src->GetWritePtr();  // Request a  pointer from the source frame.
		src_pitch = src->GetPitch();
		src_width = src->GetRowSize();
		src_height = src->GetHeight();

		DeFreq::DeFreqYUY2(plane, srcp,src_height,src_width,src_pitch, &fxPeak, &fyPeak, &sharpPeak,
			&fxPeak2, &fyPeak2, &sharpPeak2, &fxPeak3, &fyPeak3,  &sharpPeak3, &fxPeak4, &fyPeak4,  &sharpPeak4);

		if (info)
		{	// show frequency info
				int x0 = vi.width/20+1;
				int y0 = 0;
				DrawString(src, x0, y0++, "DeFreq peaks:",IsYUY2);// YUY2 compatible version of info.h added in v.0.5
				if (fx >0 || fy !=0)
				{ //first Peak
					sprintf(messagebuf,"x=%.1f y=%.1f", fxPeak, fyPeak); // bug fixed with format in v 0.5
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak > sharp)
						sprintf(messagebuf,"SHARP=%.1f", sharpPeak);
					else
						sprintf(messagebuf,"sharp=%.1f", sharpPeak);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
				if (fx2 >0 || fy2 !=0)
				{ //second Peak
					sprintf(messagebuf,"x2=%.1f y2=%.1f", fxPeak2, fyPeak2);
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak2 > sharp2)
						sprintf(messagebuf,"SHARP2=%.1f", sharpPeak2);
					else
						sprintf(messagebuf,"sharp2=%.1f", sharpPeak2);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
				if (fx3 >0 || fy3 !=0)
				{ // third Peak
					sprintf(messagebuf,"x3=%.1f y3=%.1f", fxPeak3, fyPeak3);
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak3 > sharp3)
						sprintf(messagebuf,"SHARP3=%.1f", sharpPeak3);
					else
						sprintf(messagebuf,"sharp3=%.1f", sharpPeak3);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
				if (fx4 >0 || fy4 !=0)
				{ // fourth Peak
					sprintf(messagebuf,"x4=%.1f y4=%.1f", fxPeak4, fyPeak4);
					DrawString(src, x0, y0++, messagebuf ,IsYUY2);
					if (sharpPeak4 > sharp4)
						sprintf(messagebuf,"SHARP4=%.1f", sharpPeak4);
					else
						sprintf(messagebuf,"sharp4=%.1f", sharpPeak4);
					DrawString(src, x0, y0++, messagebuf,IsYUY2);
				}
		}
	}
  // As we now are finished processing the image, we return the destination image.
	return src;
}

//-------------------------------------------------------------------------------------------

// This is the function that created the filter, when the filter has been called.
// This can be used for simple parameter checking, so it is possible to create different filters,
// based on the arguments recieved.

AVSValue __cdecl Create_DeFreq(AVSValue args, void* user_data, IScriptEnvironment* env) {
    return new DeFreq(args[0].AsClip(), // the 0th parameter is the source clip
		(float) args[1].AsFloat(10.0), // parameter fx
		(float) args[2].AsFloat(-10.0), // parameter fy
		(float) args[3].AsFloat(1.5), // parameter dx
		(float) args[4].AsFloat(2.0), // parameter dy
		(float) args[5].AsFloat(50.0), // parameter sharp
		(float) args[6].AsFloat(0), // parameter fx2
		(float) args[7].AsFloat(0), // parameter fy2
		(float) args[8].AsFloat(1.5), // parameter dx2
		(float) args[9].AsFloat(2.0), // parameter dy2
		(float) args[10].AsFloat(50.0), // parameter sharp2
		(float) args[11].AsFloat(0), // parameter fx3
		(float) args[12].AsFloat(0), // parameter fy3
		(float) args[13].AsFloat(1.5), // parameter dx3
		(float) args[14].AsFloat(2.0), // parameter dy3
		(float) args[15].AsFloat(50.0), // parameter sharp3
		(float) args[16].AsFloat(0), // parameter fx4
		(float) args[17].AsFloat(0), // parameter fy4
		(float) args[18].AsFloat(1.5), // parameter dx4
		(float) args[19].AsFloat(2.0), // parameter dy4
		(float) args[20].AsFloat(50.0), // parameter sharp4
		(float) args[21].AsFloat(0), // parameter cutx
		(float) args[22].AsFloat(0), // parameter cuty
		 args[23].AsInt(0), // parameter plane
		 args[24].AsInt(0), // parameter show
		 args[25].AsBool(false), // parameter info
		 args[26].AsBool(true), // parameter measure
		 env);  
    // Calls the constructor with the arguments provied.
}

//-------------------------------------------------------------------------------------------

// The following function is the function that actually registers the filter in AviSynth
// It is called automatically, when the plugin is loaded to see which functions this filter contains.

extern "C" __declspec(dllexport) const char* __stdcall AvisynthPluginInit2(IScriptEnvironment* env) {
    env->AddFunction("DeFreq", "c[FX]f[FY]f[DX]f[DY]f[SHARP]f[FX2]f[FY2]f[DX2]f[DY2]f[SHARP2]f[FX3]f[FY3]f[DX3]f[DY3]f[SHARP3]f[FX4]f[FY4]f[DX4]f[DY4]f[SHARP4]f[CUTX]f[CUTY]f[PLANE]i[SHOW]i[INFO]b[MEASURE]b", Create_DeFreq, 0);
    // The AddFunction has the following parameters:
    // AddFunction(Filtername , Arguments, Function to call,0);
    
    // Arguments is a string that defines the types and optional names of the arguments for you filter.
    // c - Video Clip
    // i - Integer number
    // f - Float number
    // s - String
    // b - boolean


    return "`DeFreq' DeFreq plugin";

}

